import { system, world, ItemStack, BlockComponentRandomTickEvent, BlockComponentPlayerDestroyEvent, EntityComponentTypes, EquipmentSlot, BlockComponentPlayerPlaceBeforeEvent, ItemComponentTypes, PlayerBreakBlockBeforeEvent, GameMode, Block, BlockPermutation, Dimension } from '@minecraft/server';
import { ChrysalisData } from './ChrysalisData.js';
import { Vec3 } from '../utils/Vec3.js';
import { Helper } from '../utils/Helper.js';
export class ChrysalisBlock {
	static BLOCK_TYPE = 'sf_nba:chrysalis_block';
	static COMPONENT_ID = 'sf_nba:chrysalis';
	static GROWTH_STAGE_STATE = 'sf_nba:growth_stage';
	static LOG_BLOCK_TYPES = [
		'minecraft:oak_log',
		'minecraft:spruce_log',
		'minecraft:birch_log',
		'minecraft:jungle_log',
		'minecraft:acacia_log',
		'minecraft:dark_oak_log',
		'minecraft:mangrove_log',
		'minecraft:cherry_log',
		'minecraft:pale_oak_log',
		'minecraft:stripped_oak_log',
		'minecraft:stripped_spruce_log',
		'minecraft:stripped_birch_log',
		'minecraft:stripped_jungle_log',
		'minecraft:stripped_acacia_log',
		'minecraft:stripped_dark_oak_log',
		'minecraft:stripped_mangrove_log',
		'minecraft:stripped_cherry_log',
		'minecraft:stripped_pale_oak_log'
	]
	static onChrysalisDestroyed(chrysalisData, inCreative, block, permutation, dimension) {
		if (chrysalisData) {
			const stack = new ItemStack('sf_nba:chrysalis_with_data');
			stack.setDynamicProperty(ChrysalisData.CHRYSALIS_DATA_KEY, chrysalisData);
			stack.setDynamicProperty(this.GROWTH_STAGE_STATE, permutation.getState(this.GROWTH_STAGE_STATE));
			dimension.spawnItem(stack, block.center());
			return;
		}
		if (inCreative) return;
		dimension.spawnItem(new ItemStack('sf_nba:chrysalis'), block.center());
	}
	constructor() {
		this.onRandomTick = this.onRandomTick.bind(this);
		this.onPlayerDestroy = this.onPlayerDestroy.bind(this);
		this.beforeOnPlayerPlace = this.beforeOnPlayerPlace.bind(this);
	}
	onRandomTick(event) {
		if (Math.random() < 0.75) return;
		const { GROWTH_STAGE_STATE } = ChrysalisBlock;
		const { block, dimension } = event;
		const { permutation } = block;
		const growthStage = permutation.getState(GROWTH_STAGE_STATE);
		if (growthStage === 2) {
			const butterfly = dimension.spawnEntity('sf_nba:butterfly', block.center());
			block.setType('minecraft:air');
			const nameTag = ChrysalisData.getAndDelete(dimension, block.location);
			if (!nameTag) return;
			butterfly.nameTag = nameTag;
			return;
		}
		block.setPermutation(permutation.withState(GROWTH_STAGE_STATE, growthStage + 1));
	}
	onPlayerDestroy(event) {
		const { block, dimension, player, destroyedBlockPermutation } = event;
		const { location } = block;
		const chrysalisData = ChrysalisData.getAndDelete(dimension, location);
		const inCreative = player.getGameMode() === GameMode.creative;
		const hasShears = inCreative
			? undefined
			: player.getComponent(EntityComponentTypes.Equippable)?.getEquipment(EquipmentSlot.Mainhand)?.typeId === 'minecraft:shears';
		if (!inCreative && !hasShears) return;
		ChrysalisBlock.onChrysalisDestroyed(chrysalisData, inCreative, block, destroyedBlockPermutation, dimension);
	}
	beforeOnPlayerPlace(event) {
		const { block, dimension, player } = event;
		const { location } = block;
		const chrysalisDataKey = ChrysalisData.getChrysalisDataKey(dimension, location);
		const handItem = player.getComponent(EntityComponentTypes.Equippable)?.getEquipment(EquipmentSlot.Mainhand);
		const chrysalisData = handItem?.getDynamicProperty(ChrysalisData.CHRYSALIS_DATA_KEY);
		if (!chrysalisData) {
			ChrysalisData.delete(chrysalisDataKey);
			return;
		}
		ChrysalisData.register(chrysalisDataKey, chrysalisData);
		const { GROWTH_STAGE_STATE } = ChrysalisBlock;
		event.permutationToPlace = event.permutationToPlace.withState(GROWTH_STAGE_STATE, handItem.getDynamicProperty(GROWTH_STAGE_STATE) || 0);
	}
}
class ChrysalisEvents {
	static beforeChrysalisDestroyed(event) {
		const { itemStack, dimension, block, player } = event;
		if (!itemStack || player.getGameMode() === GameMode.creative) return;
		if (!itemStack.getComponent(ItemComponentTypes.Enchantable)?.getEnchantment('silk_touch')?.level) return;
		event.cancel = true;
		const { location, permutation } = block;
		system.run(() => {
			dimension.runCommand(`setblock ${location.x} ${location.y} ${location.z} air destroy`);
			if (itemStack?.typeId !== 'minecraft:shears') return;
			const chrysalisData = ChrysalisData.getAndDelete(dimension, location);
			ChrysalisBlock.onChrysalisDestroyed(chrysalisData, false, block, permutation, dimension);
		});
	}
	static beforeChrysalisLogDestroyed(event) {
		const { block, dimension } = event;
		const { location } = block;
		const neighborCrysalis = dimension.getBlocks(Vec3.fromMinecraftVector3(location).toBlockVolume(1, 0), { includeTypes: [ChrysalisBlock.BLOCK_TYPE] }, true);
		Helper.forIterator(neighborCrysalis.getBlockLocationIterator(), (loc) => {
			const dx = Math.abs(loc.x - location.x);
			const dz = Math.abs(loc.z - location.z);
			if (dx + dz !== 1) return;
			const crysalisBlock = dimension.getBlock(loc);
			const { permutation } = crysalisBlock;
			const direction = permutation.getState('minecraft:block_face');
			const offset = {
				north: { x: 0, z: 1 },
				south: { x: 0, z: -1 },
				west: { x: 1, z: 0 },
				east: { x: -1, z: 0 }
			}[direction];
			if (!offset) return;
			if (location.x !== loc.x + offset.x || location.z !== loc.z + offset.z) return;
			ChrysalisData.delete(ChrysalisData.getChrysalisDataKey(dimension, loc));
		});
	}
}
world.beforeEvents.playerBreakBlock.subscribe(ChrysalisEvents.beforeChrysalisDestroyed, { blockTypes: [ChrysalisBlock.BLOCK_TYPE] });
world.beforeEvents.playerBreakBlock.subscribe(ChrysalisEvents.beforeChrysalisLogDestroyed, { blockTypes: ChrysalisBlock.LOG_BLOCK_TYPES });